% Optimising k-Wave Performance Example
%
% This example demonstrates how to increase the computational performance
% of k-Wave using optional input parameters and data casting.  
%
% author: Bradley Treeby
% date: 15th July 2009
% last update: 19th January 2010
%  
% This function is part of the k-Wave Toolbox (http://www.k-wave.org)
% Copyright (C) 2009, 2010 Bradley Treeby and Ben Cox

% This file is part of k-Wave. k-Wave is free software: you can
% redistribute it and/or modify it under the terms of the GNU Lesser
% General Public License as published by the Free Software Foundation,
% either version 3 of the License, or (at your option) any later version.
% 
% k-Wave is distributed in the hope that it will be useful, but WITHOUT ANY
% WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
% FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
% more details. 
% 
% You should have received a copy of the GNU Lesser General Public License
% along with k-Wave. If not, see <http://www.gnu.org/licenses/>. 

clear all;

% change scale to 2 to increase the computational time
scale = 1;

% assign the grid size and create the computational grid
Nx = 256*scale;           % number of pixels in the x (column) direction
Nz = 256*scale;           % number of pixels in the z (row) direction
x = 10e-3;          % grid size [m]
z = 10e-3;          % grid size [m]
dx = x/Nx;          % pixel width [m]
dz = z/Nz;          % pixel height [m]
kgrid = makeGrid(Nx, dx, Nz, dz);

% load the initial pressure distribution from an image and scale
p0_magnitude = 2;
source.p0 = p0_magnitude*loadImage('EXAMPLE_source_two.bmp');
source.p0 = resize(source.p0, Nx, Nz);

% define the properties of the propagation medium
medium.sound_speed = 1500;	% [m/s]
medium.density = 1000;     	% [kg/m^3]

% define a centered Cartesian circular sensor
sensor_radius = 4.5e-3;     % [m]
num_sensor_points = 100;
sensor.mask = makeCartCircle(sensor_radius, num_sensor_points);

% create the time array
[kgrid.t_array dt] = makeTime(kgrid, medium.sound_speed);

% set the input options
input_args = {'CartInterp', 'nearest', 'PlotSim', false};

% run the simulation
kspaceFirstOrder2D(kgrid, medium, source, sensor, input_args{:});

% set the input options to include data casting
input_args = {'CartInterp', 'nearest', 'PlotSim', false, 'DataCast','single'};

% run the simulation again
kspaceFirstOrder2D(kgrid, medium, source, sensor, input_args{:});